﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CrowCanyon.DBServiceApp.Classes
{
    public class DBUtilsService
    {
        /// <summary>
        /// Test Database Connection
        /// Parameters["dbConn"] - DB Connection String
        /// </summary>
        /// <param name="Parameters"></param>
        /// <returns></returns>
        public static Dictionary<string, Object> TestConnection(Dictionary<string, object> Parameters)
        {
            Dictionary<string, object> result = new Dictionary<string, object>();
            try
            {
                //Parameters["dbConn"] Get the Databse Connection string, specified in Column Setting - Database Connection
                string dbConn = Parameters["dbConn"].ToString();

                using (SqlConnection conn = new SqlConnection(dbConn))
                {
                    conn.Open();
                }

                result.Add("Data", true);

                result.Add("Result", "Success");
            }
            catch (Exception ex)
            {
                result.Add("Error", ex.ToString());
                result.Add("Result", "Error");
            }
            return result;
        }

        /// <summary>
        /// Fetch Database Table Records based on Select Query and Parameters
        /// Parameters["dbConn"] - DB Connection String
        /// Parameters["queryText"] - Select Query to fetch items from Db
        /// Parameters["isSProc"] - false
        /// Parameters["params"] - Query Parameters with Values
        /// </summary>
        /// <param name="Parameters"></param>
        /// <returns></returns>
        public static Dictionary<string, Object> SelectQuery(Dictionary<string, object> Parameters)
        {
            Dictionary<string, object> result = new Dictionary<string, object>();
            try
            {                
                //Parameters["dbConn"] Get the Databse Connection string, specified in Column Setting - Database Connection
                string dbConn = Parameters["dbConn"].ToString();

                //Parameters["queryText"] Get the Query Text for Fetch ITems from Databse, specified in Column Setting - Select Query
                string queryText = Parameters["queryText"].ToString();

                //Parameters["isSProc"] Default Value is false
                bool isSProc = (bool)Parameters["isSProc"];
                
                Dictionary<string, object> queryParams = new Dictionary<string, object>();

                //Parameters["params"] Get the Query Parameters with Value, specified in Column Setting - Query Parameters
                if (Parameters.ContainsKey("params"))
                {
                    Newtonsoft.Json.Linq.JObject objJson = Parameters["params"] as Newtonsoft.Json.Linq.JObject;
                    if (objJson != null)
                    {
                        queryParams = objJson.ToObject<Dictionary<string, object>>();
                    }
                }

                result.Add("Data", FetchData(dbConn, queryText, isSProc, queryParams));

                result.Add("Result", "Success");
            }
            catch (Exception ex)
            {
                result.Add("Error", ex.ToString());
                result.Add("Result", "Error");
            }
            return result;
        }

        public static Dictionary<string, DBTableRowsData> FetchData(string connectionString, string queryText, bool isSProc, Dictionary<string, object> queryParams)
        {
            Dictionary<string, DBTableRowsData> result = new Dictionary<string, DBTableRowsData>();
            using (SqlConnection conn = new SqlConnection(connectionString))
            {
                conn.Open();

                using (SqlCommand cmd = new SqlCommand(queryText, conn))
                {
                    cmd.CommandType = isSProc ? CommandType.StoredProcedure : CommandType.Text;
                    if (queryParams != null && queryParams.Count > 0)
                    {
                        foreach (string key in queryParams.Keys)
                        {
                            if (queryParams[key] != null)
                            {
                                cmd.Parameters.AddWithValue("@" + key, queryParams[key]);
                            }
                            else
                            {
                                cmd.Parameters.AddWithValue("@" + key, DBNull.Value);
                            }
                        }
                    }

                    DataSet ds = new DataSet();
                    SqlDataAdapter ad = new SqlDataAdapter(cmd);
                    ad.Fill(ds);

                    if (ds != null && ds.Tables != null)
                    {
                        foreach (DataTable dt in ds.Tables)
                        {
                            result.Add(dt.TableName, DBTableRowsData.ParseDataTable(dt));
                        }
                    }

                }

                return result;
            }
        }
    }

    public class DBTableRowsData
    {
        public Dictionary<string, int> Columns { get; set; }
        public List<List<object>> Values { get; set; }

        public DBTableRowsData()
        {
            this.Columns = new Dictionary<string, int>();
            this.Values = new List<List<object>>();
        }

        public static DBTableRowsData ParseDataTable(DataTable dataTable)
        {
            DBTableRowsData resultData = new DBTableRowsData();

            if (dataTable != null)
            {
                if (dataTable.Columns != null)
                {
                    int indexCol = 0;
                    foreach (DataColumn columInfo in dataTable.Columns)
                    {
                        resultData.Columns.Add(columInfo.ColumnName.Trim('[').Trim(']'), indexCol);
                        indexCol++;
                    }
                }

                if (dataTable.Rows != null)
                {
                    foreach (DataRow rowData in dataTable.Rows)
                    {
                        List<object> row = new List<object>();

                        foreach (string Column in resultData.Columns.Keys)
                        {
                            row.Add(rowData.Field<object>(Column));
                        }

                        resultData.Values.Add(row);
                    }
                }
            }

            return resultData;
        }
    }
}